<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\ConfigProfile\ConfigProfile;
use App\Entity\Device\Device;
use App\Form\ConfigProfile\ConfigProfileFileUploadType;
use App\Form\ConfigProfile\ConfigProfileFromDeviceType;
use App\Form\ConfigProfile\ConfigProfileFromExistingProfileType;
use App\Repository\ConfigProfileRepository;
use App\Repository\DeviceRepository;
use App\Service\Api\ConfigProfiles;
use App\Service\ConfigProfile\ConfigProfileService;
use Doctrine\DBAL\Exception;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;

#[Route(path: '/admin/ajax/config-profile', name: 'ajax_config_profile_')]
class ConfigProfileController extends Controller
{

	/**
	 * @param Request $request
	 * @param DeviceRepository $deviceRepository
	 * @param ConfigProfileRepository $configProfileRepository
	 * @param ManagerRegistry $managerRegistry
	 * @param ConfigProfiles $configProfilesApi
	 * @return Response
	 * @throws ClientExceptionInterface
	 * @throws RedirectionExceptionInterface
	 * @throws ServerExceptionInterface
	 * @throws TransportExceptionInterface
	 */
    #[Route(path: '/create-device', name: 'create_from_device', methods: ['POST'])]
    public function createConfigProfileFromDeviceAction(Request $request, DeviceRepository $deviceRepository,
	ConfigProfileRepository $configProfileRepository, ManagerRegistry $managerRegistry,
	ConfigProfiles $configProfilesApi): Response {

		$devices = $deviceRepository->getDevicesWithAutomaticProfile();
		$form = $this->createForm(ConfigProfileFromDeviceType::class, ['devices' => $devices]);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {

            $formData = $form->getData();
			$requestData = $request->request->all()['config_profile_from_device'];
			$deviceId = (int) ($requestData['device'] ?? null);

            try {

				$result = $configProfilesApi->createConfigProfileByDeviceId($deviceId);

				if($result->operation_result->Code === 0){

					$createdId = $result->operation_result->Description;
					$configProfile = $configProfileRepository->find($createdId);

					if($configProfile){

						$configProfile->setName($formData['label']);
						$managerRegistry->getManager()->persist($configProfile);
						$managerRegistry->getManager()->flush();

					}else{

						return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Config Profile.",
							'message' => 'Cannot update config profile name.']],
							400);

					}

				}

            }catch (\Exception $exception){

                return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Config Profile.",
					'message' => $exception->getMessage()]],
                    400);

            }

            return new JsonResponse(['code' => 200, 'message' => 'Config Profile successfully created.', 'success' => ''],
                200);
        }

        return new Response('Bad request.', 400);
    }

	/**
	 * @param Request $request
	 * @param ConfigProfileRepository $configProfileRepository
	 * @param ManagerRegistry $managerRegistry
	 * @param ConfigProfiles $configProfilesApi
	 * @return Response
	 * @throws Exception
	 */
	#[Route(path: '/create-profile', name: 'create_from_config_profile', methods: ['POST'])]
	public function createConfigProfileFromConfigProfileAction(Request $request,
															   ConfigProfileRepository $configProfileRepository,
														ManagerRegistry $managerRegistry,
														ConfigProfiles $configProfilesApi): Response {

		$configProfiles = $configProfileRepository->getProfilesToMakeCopy();
		$form = $this->createForm(ConfigProfileFromExistingProfileType::class, ['profiles' => $configProfiles]);
		$form->handleRequest($request);

		if ($form->isSubmitted()) {

			$formData = $form->getData();
			$requestData = $request->request->all()['config_profile_from_existing_profile'];
			$profileId = (int) ($requestData['profile'] ?? null);

			try {

				$result = $configProfilesApi->createConfigProfileByProfileId($profileId);

				if($result->operation_result->Code === 0){

					$createdId = $result->operation_result->Description;
					$configProfile = $configProfileRepository->find($createdId);

					if($configProfile){

						$configProfile->setName($formData['label']);
						$managerRegistry->getManager()->persist($configProfile);
						$managerRegistry->getManager()->flush();

					}else{

						return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Config Profile.",
							'message' => 'Cannot update config profile name.']],
							400);

					}

				}

			}catch (\Exception $exception){

				return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Config Profile.",
					'message' => $exception->getMessage()]],
					400);

			}

			return new JsonResponse(['code' => 200, 'message' => 'Config Profile successfully created.', 'success' => ''],
				200);
		}

		return new Response('Bad request.', 400);
	}

	/**
	 * @param Request $request
	 * @param ManagerRegistry $managerRegistry
	 * @param ConfigProfiles $configProfilesApi
	 * @return Response
	 */
	#[Route(path: '/upload', name: 'upload', methods: ['POST'])]
	public function uploadConfigProfileAction(Request $request, ManagerRegistry $managerRegistry,
											  ConfigProfileRepository $configProfileRepository,
											  ConfigProfiles $configProfilesApi, ConfigProfileService $configProfileService): Response {

		if ($request->isXmlHttpRequest()) {

			$form = $this->createForm(ConfigProfileFileUploadType::class, null, ['label' => true]);
			$form->handleRequest($request);

			if ($form->isSubmitted() && $form->isValid()) {

				$formData = $form->getData();
				$profileFileData = $form->get('profile')->getData();

				if ($profileFileData) {

					$fileContent = $profileFileData->getContent();
					$data = $configProfileService->readXlsxFile($fileContent);

					if($data){

						$uploadedData = $data->rows();

						try{

							$result = $configProfilesApi->uploadConfigProfile(json_encode($this->makeConfigProfileDataToSend($uploadedData), JSON_THROW_ON_ERROR));

							if($result->operation_result->Code === 0){

								$createdId = $result->operation_result->Description;
								$configProfile = $configProfileRepository->find($createdId);

								if($configProfile){

									$configProfile->setName($formData['label']);
									$managerRegistry->getManager()->persist($configProfile);
									$managerRegistry->getManager()->flush();

								}else{

									return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Config Profile.",
										'message' => 'Cannot update config profile name.']],
										400);

								}

							}

						}catch (\Exception $exception){

							return new JsonResponse(['code' => 400, 'message' => 'Config Profile cannot be uploaded: ',
								'errors' => ['Something bad happened.']],
								200);

						}

						return new JsonResponse([
							'code' => 200,
							//'message' => 'Config Profile "'. $configProfile->getName() .'" was generated.',
							'message' => 'Config Profile uploaded successfully.',
							'success' => ''],
							200);


					}

				}

			}

		}

		return new Response('Bad request.', 400);

	}

	/**
	 * @param Request $request
	 * @param int $configProfileId
	 * @param ManagerRegistry $managerRegistry
	 * @param ConfigProfiles $configProfilesApi
	 * @return Response
	 */
    #[Route(path: '/delete/{configProfileId}', name: 'delete', methods: ['GET'])]
    public function deleteConfigProfileAction(Request $request, int $configProfileId, ManagerRegistry $managerRegistry,
				ConfigProfiles $configProfilesApi): Response {

        if ($request->isXmlHttpRequest()) {

			$em = $managerRegistry->getManager();
            $configProfile = $managerRegistry->getRepository(ConfigProfile::class)->findOneBy(['id' => $configProfileId]);

            if($configProfile){

                try {

					$configProfile->setDeleted(true);
					$em->persist($configProfile);
					$em->flush();

                }catch (\Exception $exception){

                    return new JsonResponse(['code' => 400, 'message' => 'Config Profile cannot be deleted: ',
                        'errors' => ['Something bad happened.']],
                        200);

                }

                return new JsonResponse(['code' => 200, 'message' => 'Config Profile "'. $configProfile->getName() .'" was deleted.', 'success' => ''],
                    200);

            }

        }

        return new Response('Bad request.', 400);

    }

	#[Route(path: '/add-devices', name: 'devices_add', methods: ['POST'])]
	public function addDevicesToConfigurationProfileAction(Request $request, ConfigProfiles $configProfilesApi): Response {

		if (!$request->isXmlHttpRequest()) {
			return new Response('Bad Request', 400);
		}

		if(isset($request->request)) {

			// Request has request data ?
			/*if (!isset($request->request))
				return new JsonResponse(array('status' => 'Error'),400);*/

			$deviceIds = (!is_null($request->get('items'))) ? (array) $request->get('items') : null;
			$configProfileId = (!is_null($request->get('configProfileId'))) ? (int) $request->get('configProfileId') : null;

			if (is_null($deviceIds) || is_null($configProfileId)) {
				return new JsonResponse(['status' => 'Missing some data'], 400);
			}

			try{
				$configProfilesApi->assignDeviceToConfigProfile($configProfileId, $deviceIds);
			}catch (\Exception $exception){
				return new JsonResponse(['status' => 'Error', 'errors' => $exception->getMessage()], 200);
			}

			return new JsonResponse(['code' => 200, 'message' => 'Devices successfully inserted to the group.', 'errors' => ['errors' => ['']]],
				200);

		}

		// If we reach this point, it means that something went wrong
		return new JsonResponse(['status' => 'Error', 'message' => 'Error'],400);

	}

	#[Route(path: '/remove-device/{configProfileId}/{deviceId}', name: 'device_remove', methods: ['GET'])]
	public function deviceConfigurationProfileRemoveAction(
		int $configProfileId,
		int $deviceId,
		Request $request,
		ManagerRegistry $managerRegistry,
		ConfigProfiles $configProfilesApi
	): Response {

		if (!$request->isXmlHttpRequest()) {
			return new Response('Bad Request', 400);
		}

		if(isset($request->request)) {

			if (!isset($configProfileId) && !isset($deviceId)) {
				return new JsonResponse(['status' => 'Error'], 400);
			}

			$configProfile = $managerRegistry->getRepository(ConfigProfile::class)->findOneBy(['id' => $configProfileId]);
			$device = $managerRegistry->getRepository(Device::class)->findOneBy(['id' => $deviceId]);

			if ($configProfile && $device) {

				try{

					$configProfilesApi->unassignDeviceFromConfigProfile([$device->getId()]);

				}catch (\Exception $exception){
					return new JsonResponse(['status' => 'Error', 'errors' => $exception->getMessage()], 200);
				}

				return new JsonResponse(['code' => 200, 'message' => 'Devices successfully unassigned from the Config profile.', 'errors' => ['errors' => ['']]],
					200);

			}

		}

		// If we reach this point, it means that something went wrong
		return new JsonResponse(['status' => 'Error', 'message' => 'Error'],
			400);

	}

	private function makeConfigProfileDataToSend(array $profileData): array
	{

		$profileDataArray = [];

		foreach($profileData as $profileDataItem){

			$profileDataArray[] = ['key' => $profileDataItem[0], 'value' => (string) $profileDataItem[1]];

		}

		$profileDataReturn['Profile'] = $profileDataArray;

		return $profileDataReturn;

	}

}