<?php

/*
 * This file is part of the SgDatatablesBundle package.
 *
 * (c) stwe <https://github.com/stwe/DatatablesBundle>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Sg\DatatablesBundle\Datatable;

use Doctrine\ORM\EntityManagerInterface;
use LogicException;
use Sg\DatatablesBundle\Datatable\Column\ColumnBuilder;
use Symfony\Component\PropertyAccess\PropertyAccess;
use Symfony\Component\PropertyAccess\PropertyAccessor;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;
use Symfony\Component\Security\Core\Authorization\AuthorizationCheckerInterface;
use Symfony\Contracts\Translation\TranslatorInterface;
use Twig\Environment;

abstract class AbstractDatatable implements DatatableInterface
{
    /**
     * The AuthorizationChecker service.
     *
     * @var AuthorizationCheckerInterface
     */
    protected AuthorizationCheckerInterface $authorizationChecker;

    /**
     * The SecurityTokenStorage service.
     *
     * @var TokenStorageInterface
     */
    protected TokenStorageInterface $securityToken;

    /**
     * The Translator service.
     *
     * @var TranslatorInterface
     */
    protected TranslatorInterface $translator;

    /**
     * The Router service.
     *
     * @var RouterInterface
     */
    protected RouterInterface $router;

    /**
     * The doctrine orm entity manager service.
     *
     * @var EntityManagerInterface
     */
    protected EntityManagerInterface $em;

    /**
     * The Twig Environment.
     *
     * @var Environment
     */
    protected Environment $twig;

    /**
     * A ColumnBuilder instance.
     *
     * @var ColumnBuilder
     */
    protected ColumnBuilder $columnBuilder;

    /**
     * An Ajax instance.
     *
     * @var Ajax
     */
    protected Ajax $ajax;

    /**
     * An Options instance.
     *
     * @var Options
     */
    protected Options $options;

    /**
     * A Features instance.
     *
     * @var Features
     */
    protected Features $features;

    /**
     * A Callbacks instance.
     *
     * @var Callbacks
     */
    protected Callbacks $callbacks;

    /**
     * A Events instance.
     *
     * @var Events
     */
    protected Events $events;

    /**
     * An Extensions instance.
     *
     * @var Extensions
     */
    protected Extensions $extensions;

    /**
     * A Language instance.
     *
     * @var Language
     */
    protected Language $language;

    /**
     * The unique id for this instance.
     *
     * @var int
     */
    protected mixed $uniqueId;

    /**
     * The PropertyAccessor.
     *
     * @var PropertyAccessor
     */
    protected PropertyAccessor $accessor;

    //-------------------------------------------------

    /**
     * Used to generate unique names.
     *
     * @var array
     */
    protected static array $uniqueCounter = [];

    /**
     * @throws LogicException
     */
    public function __construct(
        AuthorizationCheckerInterface $authorizationChecker,
        TokenStorageInterface $securityToken,
        $translator,
        RouterInterface $router,
        EntityManagerInterface $em,
        Environment $twig
    ) {
        $this->validateName();

        if (isset(self::$uniqueCounter[$this->getName()])) {
            $this->uniqueId = ++self::$uniqueCounter[$this->getName()];
        } else {
            $this->uniqueId = self::$uniqueCounter[$this->getName()] = 1;
        }

        $this->authorizationChecker = $authorizationChecker;
        $this->securityToken = $securityToken;

        if (! ($translator instanceof TranslatorInterface)) {
            throw new \InvalidArgumentException(sprintf('The $translator argument of %s must be an instance of %s or %s, a %s was given.', static::class, TranslatorInterface::class, \get_class($translator)));
        }
        $this->translator = $translator;
        $this->router = $router;
        $this->em = $em;
        $this->twig = $twig;

        $metadata = $em->getClassMetadata($this->getEntity());
        $this->columnBuilder = new ColumnBuilder($metadata, $twig, $router, $this->getName(), $em);

        $this->ajax = new Ajax();
        $this->options = new Options();
        $this->features = new Features();
        $this->callbacks = new Callbacks();
        $this->events = new Events();
        $this->extensions = new Extensions();
        $this->language = new Language();

        $this->accessor = PropertyAccess::createPropertyAccessor();
    }

    //-------------------------------------------------
    // DatatableInterface
    //-------------------------------------------------

    /**
     * {@inheritdoc}
     */
    public function getLineFormatter(): ?callable
    {
        return null;
    }

    /**
     * {@inheritdoc}
     */
    public function getColumnBuilder(): ColumnBuilder
    {
        return $this->columnBuilder;
    }

    /**
     * {@inheritdoc}
     */
    public function getAjax(): Ajax
    {
        return $this->ajax;
    }

    /**
     * {@inheritdoc}
     */
    public function getOptions(): Options
    {
        return $this->options;
    }

    /**
     * {@inheritdoc}
     */
    public function getFeatures(): Features
    {
        return $this->features;
    }

    /**
     * {@inheritdoc}
     */
    public function getCallbacks(): Callbacks
    {
        return $this->callbacks;
    }

    /**
     * {@inheritdoc}
     */
    public function getEvents(): Events
    {
        return $this->events;
    }

    /**
     * {@inheritdoc}
     */
    public function getExtensions(): Extensions
    {
        return $this->extensions;
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguage(): Language
    {
        return $this->language;
    }

    /**
     * {@inheritdoc}
     */
    public function getEntityManager(): EntityManagerInterface
    {
        return $this->em;
    }

    /**
     * {@inheritdoc}
     */
    public function getOptionsArrayFromEntities(array $entities, string $keyFrom = 'id', string $valueFrom = 'name'): array
    {
        $options = [];

        foreach ($entities as $entity) {
            if (true === $this->accessor->isReadable($entity, $keyFrom) && true === $this->accessor->isReadable($entity, $valueFrom)) {
                $options[$this->accessor->getValue($entity, $keyFrom)] = $this->accessor->getValue($entity, $valueFrom);
            }
        }

        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function getUniqueId(): int
    {
        return $this->uniqueId;
    }

    /**
     * {@inheritdoc}
     */
    public function getUniqueName(): string
    {
        return $this->getName().($this->getUniqueId() > 1 ? '-'.$this->getUniqueId() : '');
    }

    //-------------------------------------------------
    // Private
    //-------------------------------------------------

    /**
     * Checks the name only contains letters, numbers, underscores or dashes.
     *
     * @throws LogicException
     */
    private function validateName(): void
    {
        $name = $this->getName();
        if (1 !== preg_match(self::NAME_REGEX, $name)) {
            throw new LogicException(sprintf('AbstractDatatable::validateName(): "%s" is invalid Datatable Name. Name can only contain letters, numbers, underscore and dashes.', $name));
        }
    }
}
