<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\ConfigProfile\ConfigProfile;
use App\Form\ConfigProfile\ConfigProfileFromDeviceType;
use App\Form\ConfigProfile\ConfigProfileFromExistingProfileType;
use App\Repository\ConfigProfileRepository;
use App\Repository\DeviceRepository;
use App\Service\Api\ConfigProfiles;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route(path: '/admin/ajax/config-profile', name: 'ajax_config_profile_')]
class ConfigProfileController extends Controller
{

	/**
	 * @param Request $request
	 * @param DeviceRepository $deviceRepository
	 * @param ConfigProfiles $configProfilesApi
	 * @return Response
	 */
    #[Route(path: '/create-device', name: 'create_from_device', methods: ['POST'])]
    public function createConfigProfileFromDeviceAction(Request $request, DeviceRepository $deviceRepository,
	ConfigProfiles $configProfilesApi): Response {

		$devices = $deviceRepository->getDevicesWithAutomaticProfile();
		$form = $this->createForm(ConfigProfileFromDeviceType::class, ['devices' => $devices]);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {

            $formData = $form->getData();
			$requestData = $request->request->all()['config_profile_from_device'];
			$deviceId = (int) ($requestData['device'] ?? null);

            try {

				$result = $configProfilesApi->createConfigProfileByDeviceId($deviceId);

            }catch (\Exception $exception){

                return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Config Profile.",
					'message' => $exception->getMessage()]],
                    400);

            }

            return new JsonResponse(['code' => 200, 'message' => 'Config Profile successfully created.', 'success' => ''],
                200);
        }

        return new Response('Bad request.', 400);
    }

	#[Route(path: '/create-profile', name: 'create_from_config_profile', methods: ['POST'])]
	public function createConfigProfileFromConfigProfileAction(Request $request, ConfigProfileRepository $configProfileRepository,
														ConfigProfiles $configProfilesApi): Response {

		$configProfiles = $configProfileRepository->getProfilesToMakeCopy();
		$form = $this->createForm(ConfigProfileFromExistingProfileType::class, ['profiles' => $configProfiles]);
		$form->handleRequest($request);

		if ($form->isSubmitted()) {

			$formData = $form->getData();
			$requestData = $request->request->all()['config_profile_from_existing_profile'];
			$profileId = (int) ($requestData['profile'] ?? null);

			try {

				$result = $configProfilesApi->createConfigProfileByProfileId($profileId);

			}catch (\Exception $exception){

				return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Config Profile.",
					'message' => $exception->getMessage()]],
					400);

			}

			return new JsonResponse(['code' => 200, 'message' => 'Config Profile successfully created.', 'success' => ''],
				200);
		}

		return new Response('Bad request.', 400);
	}

	#[Route(path: '/download/{configProfileId}', name: 'download', methods: ['GET'])]
	public function downloadConfigProfileAction(Request $request, int $configProfileId, ManagerRegistry $managerRegistry,
											  ConfigProfiles $configProfilesApi): Response {

		if ($request->isXmlHttpRequest()) {

			$em = $managerRegistry->getManager();
			$configProfile = $managerRegistry->getRepository(ConfigProfile::class)->findOneBy(['id' => $configProfileId]);

			if($configProfile){

				try {

					$result = $configProfilesApi->downloadConfigProfile($configProfile->getId());
					dump($result);
					$data = ['test'];

				}catch (\Exception $exception){

					return new JsonResponse(['code' => 400, 'message' => 'Config Profile cannot be downloaded: ',
						'errors' => ['Something bad happened.']],
						200);

				}

				return new JsonResponse([
					'code' => 200,
					'message' => 'Config Profile "'. $configProfile->getName() .'" was generated.',
					'success' => '',
					'data' => new JsonResponse($data),
					'filename' => 'configProfile_'. $configProfile->getId() .'.json'],
					200);

			}

		}

		return new Response('Bad request.', 400);

	}

	/**
	 * @param Request $request
	 * @param int $configProfileId
	 * @param ManagerRegistry $managerRegistry
	 * @param ConfigProfiles $configProfilesApi
	 * @return Response
	 */
    #[Route(path: '/delete/{configProfileId}', name: 'delete', methods: ['GET'])]
    public function deleteConfigProfileAction(Request $request, int $configProfileId, ManagerRegistry $managerRegistry,
				ConfigProfiles $configProfilesApi): Response {

        if ($request->isXmlHttpRequest()) {

			$em = $managerRegistry->getManager();
            $configProfile = $managerRegistry->getRepository(ConfigProfile::class)->findOneBy(['id' => $configProfileId]);

            if($configProfile){

                try {

					$configProfile->setDeleted(true);
					$em->persist($configProfile);
					$em->flush();

                }catch (\Exception $exception){

                    return new JsonResponse(['code' => 400, 'message' => 'Config Profile cannot be deleted: ',
                        'errors' => ['Something bad happened.']],
                        200);

                }

                return new JsonResponse(['code' => 200, 'message' => 'Config Profile "'. $configProfile->getName() .'" was deleted.', 'success' => ''],
                    200);

            }

        }

        return new Response('Bad request.', 400);

    }

}