<?php

/**
 * Uniqueweb create user command
 */

namespace Unique\UserBundle\Command;

use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Question\ChoiceQuestion;
use Symfony\Component\Console\Question\Question;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;

/**
 * Create New User.
 */
class CreateUserCommand extends Command
{
    /**
     * @var EntityManagerInterface
     */
    private EntityManagerInterface $entityManager;

    /**
     * @var UserPasswordHasherInterface
     */
    private UserPasswordHasherInterface $hasher;

    /**
     * @var string
     */
    private string $userClass;

    public function __construct(
        EntityManagerInterface $entityManager,
        UserPasswordHasherInterface $hasher,
        string $userClass)
    {

        $this->entityManager = $entityManager;
        $this->hasher = $hasher;
        $this->userClass = $userClass;

        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setName('user:create')
            ->setDescription('Application create user.')
            ->addArgument('email', InputArgument::OPTIONAL, 'Email address')
            ->addArgument('password', InputArgument::OPTIONAL, 'User password')
            ->addArgument('role', InputArgument::OPTIONAL, 'User Role')
            ->addOption('firstname', 'f', InputOption::VALUE_OPTIONAL, 'Firstname')
            ->addOption('lastname', 'l', InputOption::VALUE_OPTIONAL, 'Lastname')
            ->addOption('active', 'a', InputOption::VALUE_OPTIONAL, 'Active', 'true');
    }

    protected function interact(InputInterface $input, OutputInterface $output): void
    {
        if (!$input->getArgument('email')) {
            $question = new Question('Email Address: ', 'admin@admin.com');
            $answer = $this->getHelper('question')->ask($input, $output, $question);
            $input->setArgument('email', $answer);
        }

        if (!$input->getArgument('password')) {
            $question = new Question('Password: ', 'password');
            $answer = $this->getHelper('question')->ask($input, $output, $question);
            $input->setArgument('password', $answer);
        }

        if (!$input->getArgument('role')) {
            $question = new ChoiceQuestion('Roles: ', ['ROLE_USER', 'ROLE_ADMIN', 'ROLE_SUPER_ADMIN'], 1);
            $answer = $this->getHelper('question')->ask($input, $output, $question);
            $input->setArgument('role', $answer);
        }

    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        try{

            // Create User
            $user = (new $this->userClass())
                ->setEmail($input->getArgument('email'))
                ->setRoles([$input->getArgument('role')])
                ->setFirstname($input->getOption('firstname'))
                ->setLastname($input->getOption('lastname'))
                ->setActive(filter_var($input->getOption('active'), FILTER_VALIDATE_BOOLEAN));

            // Set Password
            $password = $this->hasher->hashPassword($user, $input->getArgument('password') ?? '123123');
            $user->setPassword($password);

            // Save
            $this->entityManager->persist($user);
            $this->entityManager->flush();

            // Output
            $output->writeln('Created User:');
            $output->writeln(sprintf('Email: <comment>%s</comment>', $user->getUserIdentifier()));
            $output->writeln(sprintf('Password: <comment>%s</comment>', $input->getArgument('password')));
            $output->writeln(sprintf('Role: <comment>%s</comment>', $input->getArgument('role')));
            $output->writeln(sprintf('Firstname: <comment>%s</comment>', $input->getOption('firstname')));
            $output->writeln(sprintf('Lastname: <comment>%s</comment>', $input->getOption('lastname')));
            $output->writeln(sprintf('Active: <comment>%s</comment>', $input->getOption('active')));

            return Command::SUCCESS;

        }catch (UniqueConstraintViolationException $e){

            $output->writeln('<error>User with given email already exists.</error>');

        }

        return Command::FAILURE;

    }
}
