#!/bin/bash
BASEDEFS="base_defs"

DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"

. "$DIR/$BASEDEFS" || { echo "Fatal: Definition file ($DIR/$BASEDEFS) not found."; exit 1; }

CERTDOMAIN=${CERTDOMAIN:-localhost}
MYCERTCN=${MYCERTCN:-$NAME sample CERT}
MYDEVCERTCN=${MYDEVCERTCN:-$NAME DEV CERT}
REGCA="myCA"
DEVCA="${SHORTNAME}_dev_CA"
REGCERTSUBJ="/C=CZ/ST=CZ/O=${SHORTNAME}/CN=${CERTDOMAIN}" # defaults for regular certificate
DEVCERTSUBJ="/C=CZ/ST=CZ/O=${SHORTNAME}/CN=${CERTDOMAIN}" # defaults for development certificate
ACTCA=""
CAKEY=""
CACERT=""
CASRL=""
SERVKEY="server.key"
SERVCERT="server.pem"
SERVCSR="server.csr"
SUBJ=""
SERNO=1000
# get data in seconds: date +%s
DATESTART=1643580460
DATEDIV=300 # one build per 5 minutes

if [[ -f "${DEVCA}.key" || ($# -eq 1 && "$1" == "DEV") ]] ; then
	echo -n "Development cert. CA key (${DEVCA}.key) found, attempting to generate new DEV certificate..."
	# generate only key, set defaults for DEV env. key
	ACTCA="$DEVCA"
	SUBJ="$DEVCERTSUBJ"
	MYDATE=`date "+%s"`
	SERNO=$(( ${SERNO}+(${MYDATE} - ${DATESTART})/${DATEDIV} ))
	if [ "$SERNO" != "${SERNO//[^0-9]}" ] ; then
		echo "failed to compute new SERNO - NaN (it is: $SERNO)" >&2
		exit 1
	fi
else
	# non-DEV CA found, set defaults
	ACTCA="$REGCA"
	SUBJ="$REGCERTSUBJ"
fi

CAKEY="${ACTCA}.key"
CACERT="${ACTCA}.pem"
CASRL="${ACTCA}.srl"
# SUB is set directly
if [[ -z "$CAKEY" || -z "$CACERT" || -z "$SUBJ" || -z "$SERVCERT" || -z "$CASRL" ]] ; then
	echo "One of CAKEY/CACERT/SUBJ/SERVCERT/CASRL is empty, exiting." >&2
	echo "   \"$CAKEY\" / \"$CACERT\" / \"$SUBJ\" / \"$SERVCERT\" / \"$CASRL\"" >&2
	exit 1
fi
#Generate CA key if missing
if [ ! -f "$CAKEY" ] ; then
	echo "CA key ($CAKEY) not found, creating new one..."
	openssl genrsa -out $CAKEY 2048
else
	echo "CA key ($CAKEY) found."
fi
#Create and self sign the Root Certificate if missing or regenerating
if [ ! -f "$CACERT"  ] ; then
	echo "CA certificate ($CACERT) not found, creating one..."
	openssl req -x509 -new -nodes -key $CAKEY -sha256 -days 3650 -subj "$SUBJ" -out "$CACERT"
else
	echo "Server certificate ($CACERT) found."
fi
#update/create serial file:
if [ ! -f "${CASRL}" ] ; then
	echo "Creating new SRL file ($CASRL), serno is $SERNO"
	echo $SERNO >${CASRL}
else
	echo "Using existing SRL file ($CASRL)"
fi
#Create certificate key:
echo "Generating CERT key: $SERVKEY..."
openssl genrsa -out "${SERVKEY}" 2048

#Create CSR:
echo "Creating CSR: $SERVCSR..."
openssl req -new -batch -sha256 -key "${SERVKEY}" -subj "$SUBJ" -out ${SERVCSR}

#Sign CSR:
echo "Signing CSR, generating server cert $SERVCERT..."
openssl x509 -req -in "$SERVCSR" -CA "$CACERT" -CAkey "$CAKEY" -out "$SERVCERT" -days 3650 -sha256
echo "Adding CA cert to server cert to have complete cert chain ($SERVCERT)."
cat "$CACERT" >>"$SERVCERT"
echo "Done"

[ "$EUID" -eq 0 ] && chown "$APPUSER" "$CASRL" "$SERVKEY" "$SERVCERT" "$SERVCSR" "$CAKEY" "$CACERT"
exit 0
